package es.redsys.rest.api.service;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import javax.net.ssl.SSLContext;
import javax.xml.namespace.QName;
import javax.xml.soap.SOAPMessage;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

import es.redsys.rest.api.constants.RestConstants;
import es.redsys.rest.api.constants.RestConstants.Environment;
import es.redsys.rest.api.constants.RestConstants.Operation;
import es.redsys.rest.api.constants.RestConstants.ResponseCode;
import es.redsys.rest.api.constants.RestConstants.ResultCode;
import es.redsys.rest.api.constants.RestConstants.TransactionType;
import es.redsys.rest.api.model.RestRequest;
import es.redsys.rest.api.model.RestResponse;
import es.redsys.rest.api.model.element.RestOperationElement;
import es.redsys.rest.api.model.message.RestResponseMessage;
import es.redsys.rest.api.utils.RestSignatureUtils;
import es.redsys.rest.api.utils.RestTransformUtils;
import es.redsys.rest.api.utils.RestUtils;

public abstract class RestService {
	private String signatureKey;
	private Dispatch<SOAPMessage> dispatch = null;
	private Environment env;
	private Operation operation;
	private boolean connected = false;
	private String serviceEndpointURL;
	
	private static ObjectMapper OBJECT_MAPPER = new ObjectMapper();
	static {
		OBJECT_MAPPER.setSerializationInclusion(Include.NON_NULL);
		OBJECT_MAPPER.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
		OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		OBJECT_MAPPER.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
	}

	public RestService(String signatureKey, Environment env, Operation operation) {
		super();
		this.signatureKey = signatureKey;
		this.env = env;
		this.operation = operation;
		this.serviceEndpointURL = null;
	}
	
	public RestService(String signatureKey, String serviceEndpointURL) {
		super();
		this.signatureKey = signatureKey;
		this.env = Environment.CUSTOM;
		this.serviceEndpointURL = serviceEndpointURL;
	}

	public RestResponse sendOperation(RestRequest message) throws Exception {
		String JSONString = null;
		String postRequest = createRequestMessage(message);
		String endpointURL = Environment.getEnvironmentEndpoint(env, operation);
		URL serviceURL = null;
		if (null != endpointURL) {
			serviceURL = new URL(endpointURL);
		} else if(null != serviceEndpointURL) {
			serviceURL = new URL(serviceEndpointURL);
		} else {
			throw new Exception("Se ha establecido un entorno personalizado pero no se ha informado de la URL de destino");
		}
		byte[] postData = postRequest.getBytes(StandardCharsets.UTF_8);
		HttpURLConnection con = null;
		try {
			con = (HttpURLConnection) serviceURL.openConnection();
			con.setDoOutput(true);
			con.setRequestMethod("POST");
			con.setRequestProperty("Cache-Control", "no-cache");
			con.setRequestProperty("Pragma", "no-cache");
			con.setConnectTimeout(Integer.parseInt(RestUtils.getPropertyValue("connectionTimeoutValue")));
			con.setReadTimeout(Integer.parseInt(RestUtils.getPropertyValue("connectionReadTimeoutValue")));
			
			if (con.getRequestMethod().equalsIgnoreCase("POST")) {
				final OutputStream outputStream = con.getOutputStream();
				outputStream.write(postData);
				outputStream.flush();
				outputStream.close();
			}
			
			int status = con.getResponseCode();
			
			if (status >= HttpURLConnection.HTTP_BAD_REQUEST){
				if (con.getErrorStream() != null) {
					System.out.println(readInputStream(con.getErrorStream()));
				}
			}
			else {
				JSONString = readInputStream(con.getInputStream());
				
				System.out.println("[INFO] Respuesta recibida:");
				
				ObjectMapper objectMapper = new ObjectMapper();
				Object json = objectMapper.readValue(JSONString.getBytes(), Object.class);
				System.out.println(objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(json));
				
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			con.disconnect();
		}
		
		return JSONString != null? createResponseMessage(JSONString): null;
		
	}
	
	protected abstract String createRequestMessage(RestRequest message) throws Exception;
	protected abstract RestResponse unMarshallResponseMessage(String message) throws Exception;
	protected abstract String createEncodedJsonMessage(RestRequest message) throws Exception;
		
	protected RestResponse createResponseMessage(String input) throws Exception {
		Boolean ok = true;
		RestResponseMessage response = new RestResponseMessage();
		Map<String, String> parsedInput = RestTransformUtils.JSONToMap(input);
		
		if (parsedInput.get("errorCode") != null) {
			response.setResult(ResultCode.KO);
			response.setApiCode(parsedInput.get("errorCode"));
			ok = false;
		}
		if (parsedInput.get("ErrorCode") != null) {
			response.setResult(ResultCode.KO);
			response.setApiCode(parsedInput.get("ErrorCode"));
			ok = false;
		}
		if (parsedInput.get("ERROR") != null ) {
			response.setResult(ResultCode.KO);
			response.setApiCode(parsedInput.get("ERROR"));
			ok = false;
		}
		
		if(ok) {
			String encodedString = parsedInput.get("Ds_MerchantParameters");
			String decodedString = RestTransformUtils.decodeJSONString(encodedString);

			System.out.println("[INFO] Ds_MerchantParameters recibidos (después de decodificar)");
			Object json = OBJECT_MAPPER.readValue(decodedString.getBytes(), Object.class);
			System.out.println(OBJECT_MAPPER.writerWithDefaultPrettyPrinter().writeValueAsString(json));
			
			RestOperationElement ISOpe = OBJECT_MAPPER.readValue(decodedString.getBytes(), RestOperationElement.class);
			String respCode = ISOpe.getResponseCode();
			String trans = ISOpe.getTransactionType();
			
			response.setOperation(ISOpe);
			
			if (!checkSignature(encodedString, parsedInput.get("Ds_Signature"))) {
				response.setResult(ResultCode.KO);
			}
			else {
				
				
				if (respCode == null && ISOpe.getPsd2() != null && ISOpe.getPsd2().trim().equals(RestConstants.RESPONSE_PSD2_TRUE)) {
					response.setResult(ResultCode.AUT);
				}
				else if (respCode == null && ISOpe.getPsd2() != null && ISOpe.getPsd2().trim().equals(RestConstants.RESPONSE_PSD2_FALSE)) {
					response.setResult(ResultCode.OK);
				}
				else if (ISOpe.requiresSCA()) {
					response.setResult(ResultCode.AUT);
				}
				else if (ResponseCode.AUTHORIZATION_OK.isCode(respCode) && 
						(TransactionType.getEnum(trans) == TransactionType.AUTHORIZATION
							|| TransactionType.getEnum(trans) == TransactionType.VALIDATION
								|| TransactionType.getEnum(trans) == TransactionType.VALIDATION_CONFIRMATION
									|| TransactionType.getEnum(trans) == TransactionType.PREAUTHORIZATION)) {
					response.setResult(ResultCode.OK);
				}
				else if (ResponseCode.CONFIRMATION_OK.isCode(respCode) && 
						TransactionType.getEnum(trans) == TransactionType.CONFIRMATION) {
					response.setResult(ResultCode.OK);
				}
				else if (ResponseCode.CANCELLATION_OK.isCode(respCode) && 
						TransactionType.getEnum(trans) == TransactionType.CANCELLATION) {
					response.setResult(ResultCode.OK);
				}
				else if (ResponseCode.REFUND_OK.isCode(respCode) && 
						(TransactionType.getEnum(trans) == TransactionType.REFUND
							|| TransactionType.getEnum(trans) == TransactionType.REFUND_WITHOUT_ORIGINAL)) {
					response.setResult(ResultCode.OK);
				}
				else if (ResponseCode.UNFINISHED.isCode(respCode) && 
							TransactionType.getEnum(trans) == TransactionType.PAYGOLD_REQUEST) {
					response.setResult(ResultCode.OK);
				}
				else {
					response.setResult(ResultCode.KO);
				}
			}
		}
		
		return response;
	}
	
	protected String readInputStream(final InputStream inputStream) throws IOException {
		final BufferedReader responseBuffer = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
		String output = null, aux = null;
		while ((output = responseBuffer.readLine()) != null) {
			if (aux == null) {
				aux = new String();
			}
			aux = aux.concat(output);
		}
		return aux;
	}
	
	protected boolean connect(Environment env, RestRequest message, Operation operation) throws Exception {
		if (null == dispatch) {
			SSLContext sslContext = null;
			
			// Service Endpoint configuration - forcing TLSv1.2
			System.setProperty(RestConstants.HTTPS_PROTOCOLS_VARIABLE, RestUtils.getPropertyValue("connectionProtocol"));
			sslContext = SSLContext.getInstance(RestUtils.getPropertyValue("connectionProtocol"));
			sslContext.init(null, null, null);
	
			// Endpoint
			String endpointURL = Environment.getEnvironmentEndpoint(env, operation);
			URL serviceURL = null;
			if (null != endpointURL) {
				serviceURL = new URL(endpointURL);
			} else if(null != serviceEndpointURL) {
				serviceURL = new URL(serviceEndpointURL);
			} else {
				throw new Exception("Se ha establecido un entorno personalizado pero no se ha informado de la URL de destino");
			}

			QName serviceQName = new QName(RestUtils.getPropertyValue("serviceTarget"), RestUtils.getPropertyValue("serviceName"));
			QName portQName = new QName(RestUtils.getPropertyValue("serviceTarget"), RestUtils.getPropertyValue("servicePort"));
			
			// Service creation
			Service service = Service.create(serviceURL, serviceQName);
	
			// Provider configuration
			dispatch = service.createDispatch(portQName, SOAPMessage.class, Service.Mode.MESSAGE);
			BindingProvider provider = (BindingProvider) dispatch;
			provider.getRequestContext().put(RestConstants.CONNECTION_TIMEOUT_LITERAL, RestUtils.getPropertyValue("connectionTimeoutValue"));
			provider.getRequestContext().put(RestConstants.READ_TIMEOUT_LITERAL, RestUtils.getPropertyValue("connectionReadTimeoutValue"));
			provider.getRequestContext().put(RestConstants.SSL_SOCKET_FACTORY_LITERAL, sslContext.getSocketFactory());
			provider.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, serviceURL.toExternalForm());
		}
		return true;
	}
	
	protected boolean checkSignature(String merchantParametersB64, String incomingSignature) throws Exception {
		String signature = RestSignatureUtils.createMerchantSignature(getSignatureKey(), merchantParametersB64, true);
		return signature.equals(incomingSignature);
	}
	
	protected String getSignatureKey(){
		return this.signatureKey;
	}
	protected Environment getEnv() {
		return env;
	}
	protected Operation getOperation() {
		return operation;
	}
}
